<?php
// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'queen_florist');
define('DB_USER', 'root');
define('DB_PASS', '');

// Security Configuration
define('JWT_SECRET', 'your-secret-key-change-this-in-production');
define('JWT_EXPIRY', 3600 * 24); // 24 hours

// Admin Configuration
define('ADMIN_USERNAME', 'admin');
define('ADMIN_PASSWORD_HASH', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi'); // password: admin123

// Error Reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);

// CORS Headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Database Connection
function getDB() {
    static $db = null;
    
    if ($db === null) {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            
            $db = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Database connection failed']);
            exit;
        }
    }
    
    return $db;
}

// Security Functions
function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

function validateInput($data, $rules) {
    $errors = [];
    
    foreach ($rules as $field => $rule) {
        $value = $data[$field] ?? '';
        
        // Required validation
        if (isset($rule['required']) && $rule['required'] && empty($value)) {
            $errors[$field] = $rule['message'] ?? ucfirst($field) . ' is required';
            continue;
        }
        
        // Skip other validations if field is empty and not required
        if (empty($value)) {
            continue;
        }
        
        // Type validation
        if (isset($rule['type'])) {
            switch ($rule['type']) {
                case 'int':
                    if (!filter_var($value, FILTER_VALIDATE_INT)) {
                        $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must be an integer';
                    }
                    break;
                case 'string':
                    if (!is_string($value)) {
                        $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must be a string';
                    }
                    break;
                case 'email':
                    if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
                        $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must be a valid email';
                    }
                    break;
            }
        }
        
        // Min/Max validation
        if (isset($rule['min']) && strlen($value) < $rule['min']) {
            $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must be at least ' . $rule['min'] . ' characters';
        }
        
        if (isset($rule['max']) && strlen($value) > $rule['max']) {
            $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must not exceed ' . $rule['max'] . ' characters';
        }
        
        // Range validation (for numbers)
        if (isset($rule['range']) && is_numeric($value)) {
            $value = (int) $value;
            if ($value < $rule['range'][0] || $value > $rule['range'][1]) {
                $errors[$field] = $rule['message'] ?? ucfirst($field) . ' must be between ' . $rule['range'][0] . ' and ' . $rule['range'][1];
            }
        }
    }
    
    return $errors;
}

// JWT Functions
function generateJWT($payload) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode($payload);
    
    $headerEncoded = base64url_encode($header);
    $payloadEncoded = base64url_encode($payload);
    
    $signature = hash_hmac('sha256', $headerEncoded . "." . $payloadEncoded, JWT_SECRET, true);
    $signatureEncoded = base64url_encode($signature);
    
    return $headerEncoded . "." . $payloadEncoded . "." . $signatureEncoded;
}

function verifyJWT($jwt) {
    if (empty($jwt)) {
        return false;
    }
    
    $tokenParts = explode('.', $jwt);
    if (count($tokenParts) != 3) {
        return false;
    }
    
    $header = base64url_decode($tokenParts[0]);
    $payload = base64url_decode($tokenParts[1]);
    $signature = $tokenParts[2];
    
    // Verify signature
    $headerEncoded = base64url_encode($header);
    $payloadEncoded = base64url_encode($payload);
    
    $expectedSignature = hash_hmac('sha256', $headerEncoded . "." . $payloadEncoded, JWT_SECRET, true);
    $signatureValid = hash_equals($signature, base64url_encode($expectedSignature));
    
    if (!$signatureValid) {
        return false;
    }
    
    // Verify expiration
    $payload = json_decode($payload, true);
    if (isset($payload['exp']) && $payload['exp'] < time()) {
        return false;
    }
    
    return $payload;
}

function base64url_encode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

function base64url_decode($data) {
    return base64_decode(str_pad(strtr($data, '-_', '+/'), strlen($data) % 4, '=', STR_PAD_RIGHT));
}

// Session Management
function createSession($username) {
    $sessionId = bin2hex(random_bytes(32));
    $expiresAt = date('Y-m-d H:i:s', time() + JWT_EXPIRY);
    
    $db = getDB();
    $stmt = $db->prepare("INSERT INTO admin_sessions (session_id, admin_id, ip_address, user_agent, expires_at) VALUES (?, ?, ?, ?, ?)");
    $stmt->execute([
        $sessionId,
        $username,
        $_SERVER['REMOTE_ADDR'] ?? '',
        $_SERVER['HTTP_USER_AGENT'] ?? '',
        $expiresAt
    ]);
    
    return $sessionId;
}

function validateSession($sessionId) {
    if (empty($sessionId)) {
        return false;
    }
    
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM admin_sessions WHERE session_id = ? AND is_active = 1 AND expires_at > NOW()");
    $stmt->execute([$sessionId]);
    $session = $stmt->fetch();
    
    if (!$session) {
        return false;
    }
    
    // Update last activity
    $updateStmt = $db->prepare("UPDATE admin_sessions SET expires_at = DATE_ADD(NOW(), INTERVAL ? SECOND) WHERE session_id = ?");
    $updateStmt->execute([JWT_EXPIRY, $sessionId]);
    
    return $session;
}

function destroySession($sessionId) {
    $db = getDB();
    $stmt = $db->prepare("UPDATE admin_sessions SET is_active = 0 WHERE session_id = ?");
    $stmt->execute([$sessionId]);
}

// Response Helper
function jsonResponse($success, $message = '', $data = null, $statusCode = 200) {
    http_response_code($statusCode);
    $response = ['success' => $success];
    
    if ($message) {
        $response['message'] = $message;
    }
    
    if ($data !== null) {
        $response['data'] = $data;
    }
    
    echo json_encode($response);
    exit;
}

// Get Authorization Header
function getAuthToken() {
    $headers = getallheaders();
    $authHeader = $headers['Authorization'] ?? '';
    
    if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
        return $matches[1];
    }
    
    return null;
}

// Require Authentication
function requireAuth() {
    $sessionId = getAuthToken();
    
    if (!$sessionId) {
        jsonResponse(false, 'Authorization required', null, 401);
    }
    
    $session = validateSession($sessionId);
    if (!$session) {
        jsonResponse(false, 'Invalid or expired session', null, 401);
    }
    
    return $session;
}
?>